#include "nrf24.hpp"
#include <cstring>
#include "freertos/FreeRTOS.h"
#include "freertos/task.h"
#include "esp_rom_sys.h"
#include "esp_log.h"

#define TAG "NRF24"

#define R_REG       0x00
#define W_REG       0x20
#define R_RX        0x61
#define W_TX        0xA0
#define FLUSH_TX    0xE1
#define FLUSH_RX    0xE2
#define NOP         0xFF

#define CONFIG      0x00
#define EN_AA       0x01
#define EN_RXADDR   0x02
#define SETUP_AW    0x03
#define RF_CH       0x05
#define RF_SETUP    0x06
#define STATUS      0x07
#define RX_ADDR_P0  0x0A
#define TX_ADDR     0x10
#define RX_PW_P0    0x11
#define FIFO_STATUS 0x17

NRF24::NRF24(spi_host_device_t host, gpio_num_t ce, gpio_num_t csn)
    : ce_pin(ce), spi_host(host)
{
    gpio_reset_pin(ce_pin);
    gpio_set_direction(ce_pin, GPIO_MODE_OUTPUT);
    gpio_set_level(ce_pin, 0);

    spi_bus_config_t bus = {};
    bus.mosi_io_num = 23;
    bus.miso_io_num = 19;
    bus.sclk_io_num = 18;
    bus.quadwp_io_num = -1;
    bus.quadhd_io_num = -1;
    bus.max_transfer_sz = 64;

    spi_bus_initialize(host, &bus, SPI_DMA_CH_AUTO);    //check czy bus zainicjowany

    spi_device_interface_config_t dev = {};
    dev.clock_speed_hz = 4 * 1000 * 1000; // 4 mhz
    dev.mode = 0;
    dev.spics_io_num = csn;
    dev.queue_size = 7;

    spi_bus_add_device(host, &dev, &spi);
}

//-------------- DO OGARNIECIA ----------------//

void NRF24::ce(bool v) {
    gpio_set_level(ce_pin, v);
}

void NRF24::writeReg(uint8_t reg, uint8_t val) {
    uint8_t tx[2] = { (uint8_t)(W_REG | (reg & 0x1F)), val };
    spi_transaction_t t = {};
    t.length = 16;
    t.tx_buffer = tx;
    spi_device_transmit(spi, &t);
}

uint8_t NRF24::readReg(uint8_t reg) {
    uint8_t tx[2] = { (uint8_t)(R_REG | (reg & 0x1F)), NOP };
    uint8_t rx[2] = {0};
    spi_transaction_t t = {};
    t.length = 16;
    t.tx_buffer = tx;
    t.rx_buffer = rx;
    spi_device_transmit(spi, &t);
    return rx[1];
}

void NRF24::writeBuf(uint8_t cmd, const uint8_t* buf, uint8_t len) {    //bufor pamieci???
    uint8_t tx[33]; 
    tx[0] = cmd;
    if (len > 32) len = 32;
    memcpy(tx + 1, buf, len);
    
    spi_transaction_t t = {};
    t.length = (len + 1) * 8;
    t.tx_buffer = tx;
    spi_device_transmit(spi, &t);
}

void NRF24::readBuf(uint8_t cmd, uint8_t* buf, uint8_t len) {
    uint8_t tx[33];
    uint8_t rx[33];
    memset(tx, NOP, sizeof(tx));
    memset(rx, 0, sizeof(rx));
    tx[0] = cmd;

    if (len > 32) len = 32;

    spi_transaction_t t = {};
    t.length = (len + 1) * 8;
    t.tx_buffer = tx;
    t.rx_buffer = rx;
    spi_device_transmit(spi, &t);
    
    memcpy(buf, rx + 1, len);
}

void NRF24::begin() {
    vTaskDelay(pdMS_TO_TICKS(100));

    uint8_t addr[5] = {0xE7, 0xE7, 0xE7, 0xE7, 0xE7}; //adresowanie
    
    ce(false);
    writeReg(CONFIG, 0x08);     
    writeReg(EN_AA, 0x01);      
    writeReg(EN_RXADDR, 0x01);  
    writeReg(SETUP_AW, 0x03);   
    writeReg(RF_CH, 76);        //wfii kanal 76 podobno poza wifi
    writeReg(RF_SETUP, 0x06);
    
    writeBuf(W_REG | TX_ADDR, addr, 5);
    writeBuf(W_REG | RX_ADDR_P0, addr, 5);
    
    writeReg(RX_PW_P0, 32);
    
    writeReg(CONFIG, 0x0E);     
    vTaskDelay(pdMS_TO_TICKS(5));
}

bool NRF24::send(const uint8_t* data) {
    ce(false);
    uint8_t config = readReg(CONFIG);
    config &= ~(1 << 0);
    writeReg(CONFIG, config);

    writeBuf(W_TX, data, 32);
    
    ce(true);
    esp_rom_delay_us(20);
    ce(false);

    for(int i = 0; i < 100; i++) {
        uint8_t s = readReg(STATUS);
        if(s & (1 << 5)) {
            writeReg(STATUS, (1 << 5));
            return true; 
        }
        if(s & (1 << 4)) {
            writeReg(STATUS, (1 << 4));
            writeBuf(FLUSH_TX, nullptr, 0);
            return false; 
        }
        vTaskDelay(pdMS_TO_TICKS(1));
    }
    return false;
}

void NRF24::startRX() {
    ce(false);
    uint8_t config = readReg(CONFIG);
    config |= (1 << 0);
    writeReg(CONFIG, config);
    
    writeReg(STATUS, 0x70);
    ce(true);
    esp_rom_delay_us(150);
}

bool NRF24::available() {
    uint8_t fifo = readReg(FIFO_STATUS);
    if ((fifo & 1) == 0) return true;
    return false;
}

void NRF24::read(uint8_t* data) {
    readBuf(R_RX, data, 32);
    writeReg(STATUS, (1 << 6));
}